﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.DataContracts;
using System.ServiceModel;
using BMS.FaultContracts;

namespace BMS.ServiceContracts
{
    /// <summary>
    /// BMS Configuration persistence operations (CRUD).
    /// </summary>
    [ServiceContract()]
    public interface IConfigurationOperations
    {
        #region Get Operations

        /// <summary>
        /// Based on the user name and domain, returns the configurations data related.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration GetUserProfile(string userName, string domainId);

        /// <summary>
        /// Returns the default user configuration data defined for the specified domain.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration GetDefaultProfile(string domainId);

        /// <summary>
        /// Gets the user profiles.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<Configuration> GetUserProfiles(string domainId);

        /// <summary>
        /// Based on the domain, returns all the configuration data for the application.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="parentSection">The parent section.</param>
        /// <param name="childSection">The section.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration GetApplicationConfiguration(string domainId, string parentSection, string childSection);

        /// <summary>
        /// Based on the domain, returns all the default configuration data for the application.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration GetDefaultApplicationConfiguration(string domainId);

        /// <summary>
        /// Gets the application configuration for home page.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="configKeys">The config keys.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Dictionary<string, DataContracts.Configuration> GetApplicationConfigurationForHomePage(string domainId, List<string> configKeys);

        #endregion

        #region Set Operations

        /// <summary>
        /// Creates, updates or deletes configuration entries for a specific user.
        /// </summary>
        /// <param name="config">The configuration data to pass to the database.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void SetUserProfile(Configuration config);

        /// <summary>
        /// Creates, updates or deletes configuration entries for the application.
        /// </summary>
        /// <param name="config">The configuration data to pass to the database.</param>
        /// <param name="parentSection">The parentSection.</param>
        /// <param name="childSection">The section data to pass to the database.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void SetApplicationConfiguration(Configuration config, string parentSection, string childSection);

        #endregion

        #region "Restore to default" Operations

        /// <summary>
        /// Restores the configuration data for a specific user and domain to defaults(default user profile).
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration ResetUserProfile(string userName, string domainId);

        /// <summary>
        /// Restores the configuration data for the application and domain to defaults(default application configuration).
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Configuration data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(ConfigValueException))]
        [FaultContract(typeof(DefaultConfigException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Configuration ResetApplicationConfiguration(string domainId);

        #endregion
    }
}
